#import "BCReader.h"
#import "StringAdditions.h"


@implementation BCReader


- (NSDictionary *)readFile:(NSString *)textFile
{
    NSDictionary *theContents;
    NSString *lineBreak;
    lineBreak = [self detectLineBreak:textFile];
    if ([textFile hasCaseInsensitivePrefix:@"#NEXUS"] || [textFile hasCaseInsensitivePrefix:@"#PAUP"])
    {
        theContents =  [self readNexusFileAndBlocks:textFile];
    }
    else if ([textFile hasCaseInsensitivePrefix:@"CLUSTAL"])
    {
        theContents =  [self readClustalFile:textFile];
    }
    else if ([textFile hasCaseInsensitivePrefix:@"Pileup"])
    {
        theContents =  [self readMSFFile:textFile];
    }
    else if ([textFile hasCaseInsensitivePrefix:@">DL"])
    {
        theContents =  [self readPirFile:textFile];
    }
    else if ([textFile hasPrefix:@">"])
    {
        theContents =  [self readFastaFile:textFile];
    }
    else if ([textFile hasPrefix:@"#"])
    {
        theContents =  [self readGDEFile:textFile];
    }
    else
    {
        theContents =  [self readPhylipFile:textFile];
    }
    [theContents setObject:lineBreak forKey:@"lineBreak"];
    return theContents;
}


- (NSDictionary *)readPhylipFile:(NSString *)textFile
{
    int i, j;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *phylipDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];
    j = 0;

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];
    //Isolate individual lines based on \r
    linesArray = (NSMutableArray*)[stringWithMacLineBreaks componentsSeparatedByString:@"\r"];

    //Remove empty lines
    for (i = 0; i < [linesArray count]; i++) {
        if ([(NSString *)[linesArray objectAtIndex:i]length] < 1 || [[linesArray objectAtIndex:i] isEqualTo:@""]) {
            [linesArray removeObjectAtIndex:i];
            i--;
        }
    }

    //Read taxon names and sequences and put them in the matrixDictionary. We can ignore the first line.
    for (i = 1; i < [linesArray count]; i++) {

        if ([[linesArray objectAtIndex:i] hasPrefix:@" "]) {	//If the taxon name is not present
            sequence = [linesArray objectAtIndex:i];
            tempSequence = (NSMutableString*)[[matrixDictionary objectForKey:[taxonArray objectAtIndex:j]] stringByAppendingString:sequence];
            [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:j]];
            j++;

            if (j == [taxonArray count]) { j = 0;}
        }
        else {							//If the taxon name precedes the sequence
            taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
            [taxonScanner scanUpToString:@" " intoString:&taxon];
            sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];
            [matrixDictionary setObject:sequence forKey:taxon];
            [taxonArray addObject:taxon];
            //NSLog(@"Adding taxon %@", taxon);
            [taxonScanner release];
        }
    }

    //Remove spaces and tabs from the sequences in the matrixDictionary
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the phylipDictionary
    [phylipDictionary setObject:matrixDictionary forKey:@"matrix"];
    [phylipDictionary setObject:taxonArray forKey:@"taxa"];

    return phylipDictionary;
}



- (NSDictionary *)readPirFile:(NSString *)textFile
{
    int i;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *pirDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];

    //Isolate individual lines based on >DL;
    linesArray = (NSMutableArray*)[textFile componentsSeparatedByString:@">DL;"];

    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = [linesArray count]-1; i >= 0 ; i--) {
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
        if ([[linesArray objectAtIndex:i]isEqualTo:@""]) {
            [linesArray removeObjectAtIndex:i];
        }
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 0; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@"\r" intoString:&taxon];
        sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];
        [matrixDictionary setObject:sequence forKey:taxon];
        [taxonArray addObject:taxon];
        [taxonScanner release];
    }


    //Remove spaces, carriage returns and * from the sequences in the matrixDictionary
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"*" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\r" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the fastaDictionary
    [pirDictionary setObject:matrixDictionary forKey:@"matrix"];
    [pirDictionary setObject:taxonArray forKey:@"taxa"];

    return pirDictionary;
}




- (NSDictionary *)readFastaFile:(NSString *)textFile
{
    int i;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *fastaDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];

    //Isolate individual lines based on >
    linesArray = (NSMutableArray*)[textFile componentsSeparatedByString:@">"];

    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = [linesArray count]; i = 0 ; i--) {
        if ([[linesArray objectAtIndex:i]isEqualTo:@""]) {
            [linesArray removeObjectAtIndex:i];
        }
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 1; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@"\r" intoString:&taxon];
        sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];
        [matrixDictionary setObject:sequence forKey:taxon];
        [taxonArray addObject:taxon];
        [taxonScanner release];
    }


    //Remove spaces, carriage returns and tabs from the sequences in the matrixDictionary
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\r" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the fastaDictionary
    [fastaDictionary setObject:matrixDictionary forKey:@"matrix"];
    [fastaDictionary setObject:taxonArray forKey:@"taxa"];

    return fastaDictionary;
}


- (NSDictionary *)readGDEFile:(NSString *)textFile
{
    int i;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *gdeDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];

    //Isolate individual lines based on #
    linesArray = (NSMutableArray*)[stringWithMacLineBreaks componentsSeparatedByString:@"#"];

    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = [linesArray count]; i = 0 ; i--) {
        if ([[linesArray objectAtIndex:i]isEqualTo:@""]) {
            [linesArray removeObjectAtIndex:i];
        }
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 1; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@"\r" intoString:&taxon];
        sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];
        [matrixDictionary setObject:sequence forKey:taxon];
        [taxonArray addObject:taxon];
        [taxonScanner release];
    }


    //Remove spaces, carriage returns and tabs from the sequences in the matrixDictionary
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\r" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the fastaDictionary
    [gdeDictionary setObject:matrixDictionary forKey:@"matrix"];
    [gdeDictionary setObject:taxonArray forKey:@"taxa"];

    return gdeDictionary;
}



- (NSDictionary *)readClustalFile:(NSString *)textFile
{
    int i;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *clustalDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];
    //Isolate individual lines based on \r
    linesArray = (NSMutableArray *)[stringWithMacLineBreaks componentsSeparatedByString:@"\r"];


    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = 0; i < [linesArray count]; i++) {
        if ([[linesArray objectAtIndex:i]isEqualTo:@""] || [[linesArray objectAtIndex:i]hasPrefix:@"  "]) {
            [linesArray removeObjectAtIndex:i];
            i--;
        }
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 1; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@" " intoString:&taxon];
        sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];

        if ([matrixDictionary objectForKey:taxon]) {	//If the taxon already exists
            tempSequence = (NSMutableString*)[[matrixDictionary objectForKey:taxon] stringByAppendingString:sequence];
            [matrixDictionary setObject:tempSequence forKey:taxon];
        }
        else {						//If the taxon does not yet exist
            [matrixDictionary setObject:sequence forKey:taxon];
            [taxonArray addObject:taxon];
        }
        [taxonScanner release];
    }


    //Remove spaces and tabs from the sequences in the matrixDictionary
    //taxonArray = [matrixDictionary allKeys];
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the clustalDictionary
    [clustalDictionary setObject:matrixDictionary forKey:@"matrix"];
    [clustalDictionary setObject:taxonArray forKey:@"taxa"];

    return clustalDictionary;
}


- (NSDictionary *)readMSFFile:(NSString *)textFile
{
    int i;
    NSScanner *taxonScanner;
    NSString *stringWithMacLineBreaks, *trimmedString;
    NSMutableString *taxon, *sequence, *tempSequence;
    NSMutableArray *linesArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *MSFDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];

    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:textFile];
    //Trim the first part of the string
    trimmedString = [[stringWithMacLineBreaks componentsSeparatedByString:@"//"]objectAtIndex:1];

    //Isolate individual lines based on \r
    linesArray = (NSMutableArray *)[trimmedString componentsSeparatedByString:@"\r"];


    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = [linesArray count]-1; i >= 0 ; i--) {
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
        if ([[linesArray objectAtIndex:i]isEqualTo:@""] || [[linesArray objectAtIndex:i]hasPrefix:@"  "]) {
            [linesArray removeObjectAtIndex:i];
        }
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 0; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@" " intoString:&taxon];
        sequence = (NSMutableString*)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];

        if ([matrixDictionary objectForKey:taxon]) {	//If the taxon already exists
            tempSequence = (NSMutableString*)[[matrixDictionary objectForKey:taxon] stringByAppendingString:sequence];
            [matrixDictionary setObject:tempSequence forKey:taxon];

        }
        else {						//If the taxon does not yet exist
            [matrixDictionary setObject:sequence forKey:taxon];
            [taxonArray addObject:taxon];
        }
        [taxonScanner release];
    }


    //Remove spaces and tabs from the sequences in the matrixDictionary
    //taxonArray = [matrixDictionary allKeys];
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the clustalDictionary
    [MSFDictionary setObject:matrixDictionary forKey:@"matrix"];
    [MSFDictionary setObject:taxonArray forKey:@"taxa"];

    return MSFDictionary;
}


- (NSDictionary *)readNexusFile:(NSString *)textFile
{
    int i;
    NSScanner *matrixScanner, *taxonScanner, *commentScanner, *treeScanner;
    NSString *comments = @"";
    NSString *treeString = @"";
    NSString *stringWithMacLineBreaks;
    NSMutableString *taxon, *sequence, *tempSequence, *matrix, *bracketsString;
    NSMutableArray *linesArray, *bracketsArray;
    NSMutableDictionary *matrixDictionary = [NSMutableDictionary dictionary];
    NSMutableDictionary *nexusDictionary = [NSMutableDictionary dictionary];
    NSMutableArray *taxonArray = [NSMutableArray arrayWithCapacity:10];
    NSMutableArray *treesArray = [NSMutableArray arrayWithCapacity:10];

    //Extract the Nexus file's main comments
    commentScanner = [[NSScanner alloc] initWithString:textFile];
    [commentScanner scanUpToString:@"[!" intoString:nil];
    [commentScanner scanString:@"[!" intoString:nil];
    [commentScanner scanUpToString:@"]" intoString:&comments];
    [commentScanner release];
    //Put the comments in the nexusDictionary
    if (![comments isEqualTo:@""]) {
        [nexusDictionary setObject:comments forKey:@"comments"];
    }


    //Check whether a treeblock exists and extract it
    if ([textFile rangeOfString:@"begin trees;" options:NSCaseInsensitiveSearch||NSBackwardsSearch].length != 0) {
        treeScanner = [[NSScanner alloc] initWithString:textFile];
        [treeScanner scanUpToString:@"begin trees;" intoString:nil];
        [treeScanner scanString:@"begin trees;" intoString:nil];
        [treeScanner scanUpToString:@"end;" intoString:&treeString];
        [treeScanner release];
        treesArray = [NSMutableArray arrayWithArray:[treeString componentsSeparatedByString:@";"]];
        //Put the trees in the treesArray
        if ([treesArray count] > 0) {
            [treesArray removeObjectAtIndex:0];
            [treesArray removeObjectAtIndex:[treesArray count]-1];
            //NSLog(@"%@", treesArray);
            [nexusDictionary setObject:treesArray forKey:@"trees"];
        }
    }


    //Remove other comments (in sequences) between the []
    bracketsString = [NSMutableString stringWithString:textFile];
    [bracketsString replaceOccurrencesOfString:@"[" withString:@"!!!*" options:NULL range:NSMakeRange(0, [bracketsString length])];
    [bracketsString replaceOccurrencesOfString:@"]" withString:@"!!!" options:NULL range:NSMakeRange(0, [bracketsString length])];
    bracketsArray = (NSMutableArray *)[bracketsString componentsSeparatedByString:@"!!!"];
    for (i = 0; i < [bracketsArray count]; i++) {
        if ([[bracketsArray objectAtIndex:i]hasPrefix:@"*"]){
            [bracketsArray removeObjectAtIndex:i];
        }
    }
    matrix = (NSMutableString *)[bracketsArray componentsJoinedByString:@""];


    //Isolate the matrix
    matrixScanner = [[NSScanner alloc] initWithString:matrix];
    [matrixScanner scanUpToString:@"matrix" intoString:nil];
    [matrixScanner scanUpToString:@";" intoString:&matrix];
    [matrixScanner release];


    //Convert all line breaks to Mac line breaks (\r)
    stringWithMacLineBreaks = [self convertLineBreaksToMac:matrix];
    //Isolate individual lines based on \r
    linesArray = (NSMutableArray *)[stringWithMacLineBreaks componentsSeparatedByString:@"\r"];


    //Trim lines from surrounding whitespaces and remove empty lines
    for (i = 0; i < [linesArray count]; i++) {
        [linesArray insertObject:[[linesArray objectAtIndex:i]stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]] atIndex:i];
        [linesArray removeObjectAtIndex:i+1];
        if ([[linesArray objectAtIndex:i]isEqualTo:@""]) {
            [linesArray removeObjectAtIndex:i];
            i--;
        }
    }


    //Read taxon names and sequences and put them in the matrixDictionary
    for (i = 1; i < [linesArray count]; i++) {
        taxonScanner = [[NSScanner alloc] initWithString:[linesArray objectAtIndex:i]];
        [taxonScanner scanUpToString:@" " intoString:&taxon];
        sequence = (NSMutableString *)[[linesArray objectAtIndex:i]substringFromIndex:[taxonScanner scanLocation]];

        if ([matrixDictionary objectForKey:taxon]) {	//If the taxon already exists
            tempSequence = [NSMutableString stringWithString:[[matrixDictionary objectForKey:taxon] stringByAppendingString:sequence]];
            [matrixDictionary setObject:tempSequence forKey:taxon];

        }
        else {						//If the taxon does not yet exist
            [matrixDictionary setObject:sequence forKey:taxon];
            [taxonArray addObject:taxon]; //Put the taxon name in the taxonArray
        }
        [taxonScanner release];
    }


    //Remove spaces and tabs from the sequences in the matrixDictionary
    for (i = 0; i < [taxonArray count]; i++) {
        tempSequence = [NSMutableString stringWithString:[matrixDictionary objectForKey:[taxonArray objectAtIndex:i]]];
        [tempSequence replaceOccurrencesOfString:@" " withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [tempSequence replaceOccurrencesOfString:@"\t" withString:@"" options:NULL range:NSMakeRange(0, [tempSequence length])];
        [matrixDictionary setObject:tempSequence forKey:[taxonArray objectAtIndex:i]];
    }

    //Put the matrixDictionary and the taxa Array in the nexusDictionary
    [nexusDictionary setObject:matrixDictionary forKey:@"matrix"];
    [nexusDictionary setObject:taxonArray forKey:@"taxa"];

    return nexusDictionary;
}


- (NSDictionary *)readNexusFileAndBlocks:(NSString *)textFile
{
    int i;
    NSMutableDictionary *dict;
    NSScanner *scanner;
    NSArray *blocksArray;
    NSString *blockName, *blockContents;
    NSMutableDictionary *blocksDictionary = [NSMutableDictionary dictionary];
    dict = (NSMutableDictionary*)[self readNexusFile:textFile];

    //Separate every block
    blocksArray = [textFile componentsSeparatedByString:@"BEGIN "];
    //Use the block name as the key and the contents as the value
    for (i = 1; i < [blocksArray count]; i++) {
        scanner = [[NSScanner alloc] initWithString:[blocksArray objectAtIndex:i]];
        [scanner scanUpToString:@";" intoString:&blockName];
        [scanner scanUpToString:@"end;" intoString:&blockContents];
        [scanner release];
        [blocksDictionary setObject:blockContents forKey:blockName];
    }
    [dict setObject:blocksDictionary forKey:@"blocks"];
    return dict;
}


- (NSMutableString *)convertLineBreaksToMac:(NSString *)textFile
{
    // \r\n (Windows) becomes \r\r - \n (Unix) becomes \r
    NSMutableString *theString = [NSMutableString stringWithString:textFile];
    [theString replaceOccurrencesOfString:@"\r\n" withString:@"\r" options:NULL range:NSMakeRange(0, [theString length])];
    [theString replaceOccurrencesOfString:@"\n" withString:@"\r" options:NULL range:NSMakeRange(0, [theString length])];
    return theString;
}


- (NSString *)detectLineBreak:(NSString *)srcNStr
{
    // search for dos
    if ([srcNStr rangeOfString: @"\r\n"].location != NSNotFound)
        return @"\r\n";
    // search for mac
    else if ([srcNStr rangeOfString: @"\n"].location != NSNotFound)
        return @"\n";
    // search for unix
    else if ([srcNStr rangeOfString: @"\n"].location != NSNotFound)
        return @"\n";
    // otherwise unknown
    else
        return @"\r";
}




// Memory management

- (id)init
{
    if (self = [super init]) {
    }
    return self;
}


- (void)dealloc
{
    [super dealloc];
}



@end
